<?php
/**
 * CakeManager (http://cakemanager.org)
 * Copyright (c) http://cakemanager.org
 *
 * Licensed under The MIT License
 * For full copyright and license information, please see the LICENSE.txt
 * Redistributions of files must retain the above copyright notice.
 *
 * @copyright     Copyright (c) http://cakemanager.org
 * @link          http://cakemanager.org CakeManager Project
 * @since         1.0
 * @license       http://www.opensource.org/licenses/mit-license.php MIT License
 */
namespace Utils\View\Helper;

/**
 * MenuBuilder Interface
 *
 * This interface contains several methods who are used to build up a menu.
 *
 * ### Building up an menu
 *
 * Before the menu items will be generated the `beforeMenu` method will be called.
 * This method can be used to add an `<ul>`.
 *
 * Menu items are generated with the `item` method. This method receives the item
 * itself and some options.
 *
 * Subitems are generated by the `subItem` method. Note that when an item contains
 * subitems, the `beforeSubItem` and `afterSubItem` method will be raised.
 * This can be helpfull when you want to add an extra `<ul>` and `</ul`>
 *
 * When closing the menu the `afterMenu` method will be called. This is useful
 * to add an `</ul>` to your html.
 *
 */
interface MenuBuilderInterface
{

    /**
     * beforeMenu
     *
     * Method before the menu has been build.
     *
     * @param array $menu The menu items.
     * @param array $options Options.
     * @return string
     */
    public function beforeMenu($menu = [], $options = []);

    /**
     * beforeItem
     *
     * Method before an item has been build.
     *
     * @param array $item The menu item.
     * @param array $options Options.
     * @return string
     */
    public function beforeItem($item = [], $options = []);

    /**
     * item
     *
     * Method to build an menu item.
     *
     * @param array $item The menu item.
     * @param array $options Options.
     * @return string
     */
    public function item($item = [], $options = []);

    /**
     * afterItem
     *
     * Method after an item has been build.
     *
     * @param array $item The menu item.
     * @param array $options Options.
     * @return string
     */
    public function afterItem($item = [], $options = []);

    /**
     * afterSubItem
     *
     * Method before a submenu item has been build.
     *
     * @param array $item The menu items.
     * @param array $options Options.
     * @return string
     */
    public function beforeSubItem($item = [], $options = []);

    /**
     * item
     *
     * Method to build an submenu item.
     *
     * @param array $item The menu item.
     * @param array $options Options.
     * @return string
     */
    public function subItem($item = [], $options = []);

    /**
     * afterSubItem
     *
     * Method after a submenu item has been build.
     *
     * @param array $item The menu items.
     * @param array $options Options.
     * @return string
     */
    public function afterSubItem($item = [], $options = []);

    /**
     * afterMenu
     *
     * Method after the menu has been build.
     *
     * @param array $menu The menu items.
     * @param array $options Options.
     * @return string
     */
    public function afterMenu($menu = [], $options = []);
}
